package com.agilex.healthcare.veteranappointment.dataservice;

import com.agilex.healthcare.directscheduling.domain.PatientRequestLimit;
import com.agilex.healthcare.mobilehealthplatform.domain.DataIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.serviceregistry.ScopeFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.veteranappointment.datalayer.appointment.AppointmentRequestDataLayer;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequest;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequestMessage;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequestMessages;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequests;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.veteranappointment.exception.InvalidAppointmentRequestException;
import com.agilex.healthcare.veteranappointment.exception.InvalidAppointmentRequestMessageException;
import com.agilex.healthcare.veteranappointment.utils.VamfUserTransformer;
import com.agilex.healthcare.veteranappointment.validator.AppointmentRequestMessageValidator;
import com.agilex.healthcare.veteranappointment.validator.AppointmentRequestMessageValidatorFactory;
import com.agilex.healthcare.veteranappointment.validator.AppointmentRequestValidator;
import com.agilex.healthcare.veteranappointment.validator.AppointmentRequestValidatorFactory;
import com.agilex.healthcare.veteranappointment.validator.ValidationError;
import com.agilex.healthcare.veteranappointment.validator.ValidationResult;
import gov.va.vamf.mdws.client.v1.mobilehealthplatform.domain.MhpUser;
import gov.va.vamf.scheduling.direct.InstitutionService;
import gov.va.vamf.scheduling.direct.datalayer.appointment.PatientIdentifiersDataService;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.ClinicalServiceService;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.RequestEligibilityCriteriaService;
import gov.va.vamf.scheduling.direct.datalayer.facility.CustomFriendlyTextService;
import gov.va.vamf.scheduling.direct.domain.ClinicalService;
import gov.va.vamf.scheduling.direct.domain.CoreSetting;
import gov.va.vamf.scheduling.direct.domain.CustomFriendlyText;
import gov.va.vamf.scheduling.direct.domain.CustomFriendlyTexts;
import gov.va.vamf.scheduling.direct.domain.CustomRequestSetting;
import gov.va.vamf.scheduling.direct.domain.CustomRequestSettings;
import gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria;
import gov.va.vamf.scheduling.direct.eligibilitychecker.PatientClinicalServicesEligibilityChecker;
import gov.va.vamf.scheduling.facility.timezone.FacilityTimezoneService;
import gov.va.vamf.security.v1.VamfUser;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

@Component
public class AppointmentRequestDataService {

	@Resource
	AppointmentRequestDataLayer appointmentRequestDataLayer;

	@Resource
	ClinicalServiceService clinicalServiceService;

	@Resource
	PatientClinicalServicesEligibilityChecker patientClinicalServicesEligibilityChecker;

	@Resource
	RequestEligibilityCriteriaService requestEligibilityCriteriaService;

	@Resource
	InstitutionService institutionService;

	@Resource
	FacilityTimezoneService facilityTimezoneService;

	@Resource
	PatientIdentifiersDataService patientIdentifiersDataService;

	@Resource
	private CustomFriendlyTextService customFriendlyTextService;

	private static final int EarliestTimeVeteranCanRequestAppointmentInHours = 96;
	private static final int FarthestTimeVeteranCanRequestAppointmentInDays = 120;
    private static final int messageLimitPerSender = 2;

	private static final String STATUS_NOT_FOUND_ERROR_MSG = "The appointment request is missing a valid status.";
	private static final String INVALID_STATUS_ERROR_MSG = "The requested appointment request status is invalid.";
    private static final String APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_FOR_CANCELLED_APPOINTMENT = "Appointment request messages are not allowed for cancelled appointment requests.";
    private static final String APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_AFTER_APPOINTMENT_DATE_PASSED = "Appointment request messages are not allowed after appointment date has passed.";
    private static final String APPOINTMENT_REQUEST_MSG_LIMIT_REACHED = "Maximum allowed number of messages for this appointment request reached.";
    private static final String VAR_SYSTEM_IDENTIFIER = "var";
    private static final String ROLE_STAFF = "ROLE_STAFF";

	private static final String EXPRESS_CARE_ID = "CR1";
	public static final String COMMUNITY_CARE_TYPE = "CC";

	public VARAppointmentRequests getPatientAppointmentRequests(PatientIdentifier patientIdentifier,
																DateFilter dateFilter, ScopeFilter scopeFilter) {
		PatientIdentifiers targetIdentifiers = patientIdentifiersDataService.getCorrespondIdentifiers(patientIdentifier);
		targetIdentifiers = patientIdentifiersDataService.filterForEdipiAndIcn(targetIdentifiers);
        VARAppointmentRequests appointmentRequests =
			readAppointmentRequestsFromRouterResponseAndSortDescendingByLastUpdatedDate(
				appointmentRequestDataLayer.fetchAppointmentRequestsByPatient(targetIdentifiers,
					dateFilter), patientIdentifier);

		updateAppointmentRequestsWithMongoFriendlyName(appointmentRequests);

		return appointmentRequests;
	}
	public int getSubmittedRequestCountByApptType(PatientIdentifier patientIdentifier, DateFilter dateFilter,
												  String typeOfCareId, String institutionCode) {

		return appointmentRequestDataLayer.getSubmittedRequestCountByApptType(
			patientIdentifier.getUniqueId(),
			dateFilter,
			typeOfCareId,
			institutionCode
		);
	}

	private VARAppointmentRequests readAppointmentRequestsFromRouterResponseAndSortDescendingByLastUpdatedDate(
		VARAppointmentRequests appointmentRequests, PatientIdentifier patientIdentifier) {

		// TODO: When SM updated to support ICN, revert this back to pulling patient id from appointmentRequest
        for (VARAppointmentRequest appointmentRequest : appointmentRequests){
            appointmentRequest.setPatientIdentifier(patientIdentifier);

			appointmentRequest.setDataIdentifier(new DataIdentifier(VAR_SYSTEM_IDENTIFIER,
				appointmentRequest.getDataIdentifier().getUniqueId()));
        }
		return appointmentRequests;
	}

	public VARAppointmentRequest getPatientAppointmentRequest(PatientIdentifier patientIdentifier,
															  DataIdentifier dataIdentifier) {
        PatientIdentifiers identifiers = patientIdentifiersDataService.getCorrespondIdentifiers(patientIdentifier);

		VARAppointmentRequest appointmentRequest =
			appointmentRequestDataLayer.fetchAppointmentRequest(identifiers,
				dataIdentifier.getUniqueId());

        appointmentRequest.setPatientIdentifier(patientIdentifier);

        appointmentRequest.setDataIdentifier(new DataIdentifier(VAR_SYSTEM_IDENTIFIER,
			appointmentRequest.getDataIdentifier().getUniqueId()));

		return appointmentRequest;
	}

	public VARAppointmentRequestMessages fetchAppointmentRequestMessages(PatientIdentifier patientIdentifier,
																		 DataIdentifier dataIdentifier) {
		PatientIdentifiers identifiers =
				patientIdentifiersDataService.getCorrespondIdentifiers(patientIdentifier);

		VARAppointmentRequestMessages messages =
			appointmentRequestDataLayer.fetchAppointmentRequestMessages(identifiers,
				dataIdentifier.getUniqueId());

        for(VARAppointmentRequestMessage appointmentRequestMessage : messages){
            appointmentRequestMessage.setPatientIdentifier(patientIdentifier);
            appointmentRequestMessage.setDataIdentifier(dataIdentifier);
        }
		messages = readAppointmentRequestMessagesFromRouterResponseAndSortDescendingMessageDateTime(messages);
		return messages;
	}

	private VARAppointmentRequestMessages readAppointmentRequestMessagesFromRouterResponseAndSortDescendingMessageDateTime(
		VARAppointmentRequestMessages messages) {

		messages.sortDescending("messageDateTime");
		return messages;
	}

	public VARAppointmentRequest saveAppointmentRequest(VARAppointmentRequest appointmentRequest, String haMode, boolean fullValidation, VamfUser vamfUser) {
		Patient patient = VamfUserTransformer.getPatient(vamfUser);

		// Check if EDIPI was passed in the security context. If not, perform MVI lookup for EDIPI
		validateEdipiPatientIdentifier(patient, appointmentRequest);

		patient.setTextMessagingAllowed(appointmentRequest.isTextMessagingAllowed());
		patient.setTextMessagingPhoneNumber(appointmentRequest.getTextMessagingPhoneNumber());

		final String requestToSaveId = appointmentRequest.getAppointmentRequestId();
		if( requestToSaveId != null ) {
			PatientIdentifiers identifiers = new PatientIdentifiers(appointmentRequest.getPatientIdentifier());
			identifiers.add(patient.getPatientIdentifier());
			final VARAppointmentRequest existingRequest = appointmentRequestDataLayer.fetchAppointmentRequest(identifiers, requestToSaveId);
			patient.setPatientIdentifier(existingRequest.getPatientIdentifier());
		}

		appointmentRequest.setPatient(patient);
		if (appointmentRequest.getParentRequest() != null) {
			appointmentRequest.getParentRequest().setPatient(patient);
		}

		String typeOfCareId = appointmentRequest.getTypeOfCareId();
		ClinicalService clinicalService = clinicalServiceService.fetchById(typeOfCareId);
		if(!appointmentRequest.getStatus().equals(AppointmentRequestStatus.CANCELLED.getName())) {
			validateAppointmentRequest(appointmentRequest, fullValidation, clinicalService);
		}
        addSurrogate(appointmentRequest, vamfUser);

        updateAppointmentRequestsWithMongoFriendlyName(Arrays.asList(appointmentRequest));

        VARAppointmentRequest result = appointmentRequestDataLayer.saveAppointmentRequest(appointmentRequest);
        result.setPatientIdentifier(appointmentRequest.getPatientIdentifier());

		return result;
	}

	public VARAppointmentRequestMessage saveAppointmentRequestMessage(VARAppointmentRequestMessage appointmentRequestMessage, VamfUser vamfUser) {
		PatientIdentifier currentIdentifier = appointmentRequestMessage.getPatientIdentifier();
		appointmentRequestMessage.setPatientIdentifier(currentIdentifier);

		VARAppointmentRequest associatedAppointmentRequest =
			getPatientAppointmentRequest(currentIdentifier,
				appointmentRequestMessage.getDataIdentifier());

        appointmentRequestMessage.setMessageDateTime(getFacilityDt(associatedAppointmentRequest));
        validateAppointmentRequestMessage(appointmentRequestMessage, associatedAppointmentRequest);

        addSurrogate(appointmentRequestMessage, vamfUser);
        boolean isProvider = doesUserHasStaffRole(vamfUser);
		PatientIdentifiers identifiers = patientIdentifiersDataService.getCorrespondIdentifiers(currentIdentifier);
        VARAppointmentRequestMessage result = appointmentRequestDataLayer.saveAppointmentRequestMessage(identifiers, appointmentRequestMessage, isProvider);

		// Set the PatientIdentifier to current (vs the EDIPI identifier used for saving)
		result.setPatientIdentifier(currentIdentifier);

		return result;
	}

	public Date getFacilityDt(VARAppointmentRequest appointmentRequest) {
		String siteCode = appointmentRequest.getFacility().getParentSiteCode();
		return facilityTimezoneService.getFacilityAdjustedDate(siteCode);
	}

    private void validateAppointmentRequestMessage(VARAppointmentRequestMessage appointmentRequestMessage,
												   VARAppointmentRequest associatedAppointmentRequest) {

        AppointmentRequestMessageValidator validator =
			AppointmentRequestMessageValidatorFactory.createAppointmentRequestMessageValidator();

        if (validator != null) {
            ValidationResult<VARAppointmentRequestMessage> validationResult =
				validator.validate(appointmentRequestMessage);

            if (!validationResult.isValid()) {
                throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
            }
        }

        throwAnErrorIfAppointmentRequestIsCancelled(associatedAppointmentRequest, appointmentRequestMessage);
        throwAnErrorIfAppointmentRequestDateHasPassed(associatedAppointmentRequest, appointmentRequestMessage);
        throwAnErrorIfMessageLimitReached(associatedAppointmentRequest, appointmentRequestMessage);
    }

    public VARAppointmentRequest updateAppointmentMessageFlag(String appointmentRequestId, PatientIdentifier patientIdentifier, DataIdentifier dataIdentifier, VamfUser vamfUser){
		boolean isProvider = doesUserHasStaffRole(vamfUser);
		PatientIdentifiers identifiers = patientIdentifiersDataService.getCorrespondIdentifiers(patientIdentifier);
        VARAppointmentRequest appointmentRequest = appointmentRequestDataLayer.updateMessageFlag(identifiers, appointmentRequestId, isProvider);
        addSurrogate(appointmentRequest, vamfUser);
		return appointmentRequest;
	}

	private void validateAppointmentRequest(VARAppointmentRequest appointmentRequest, boolean fullValidation, ClinicalService clinicalService) {
		String typeOfCareId = appointmentRequest.getTypeOfCareId();
		int earliestRequestInHours = typeOfCareId.equals(EXPRESS_CARE_ID) ? 0 : EarliestTimeVeteranCanRequestAppointmentInHours;

		AppointmentRequestValidator validator = AppointmentRequestValidatorFactory
			.createAppointmentRequestValidator(earliestRequestInHours, FarthestTimeVeteranCanRequestAppointmentInDays);

		if (validator != null) {
			ValidationResult<VARAppointmentRequest> validationResult = validator.validate(appointmentRequest,
				fullValidation, clinicalService);
			if (!validationResult.isValid()) {
				throw new InvalidAppointmentRequestException(validationResult.getErrors());
			}
		}

		validateStatus(appointmentRequest);

		if(typeOfCareId.equals(EXPRESS_CARE_ID) && appointmentRequest.getDataIdentifier().getUniqueId() == null) {
			validateExpressCareSubmitWindow(appointmentRequest);
		}
	}

	private void validateExpressCareSubmitWindow(VARAppointmentRequest appointmentRequest) {

		CustomRequestSetting customRequestSetting = findCustomRequestSetting(appointmentRequest);

		if(customRequestSetting != null) {
			String facilityId = appointmentRequest.getFacility().getParentSiteCode();

			PatientClinicalServicesEligibilityChecker.ExpressCareTimeValidityAndDetail supported =
					patientClinicalServicesEligibilityChecker.isCustomRequestSupportedNow(facilityId, customRequestSetting);

			if (supported.getValidity() != PatientClinicalServicesEligibilityChecker.ExpressCareTimeValidity.VALID) {
				String errorMessage = supported.formatMessage();
				ValidationResult<VARAppointmentRequest> validationResult =
						createValidationError(appointmentRequest, "Submission expired", errorMessage);
				throw new InvalidAppointmentRequestException(validationResult.getErrors());
			}
		}
	}

	private CustomRequestSetting findCustomRequestSetting(VARAppointmentRequest appointmentRequest) {

		String facilityId = appointmentRequest.getFacility().getFacilityCode();

		RequestEligibilityCriteria requestEligibilityCriteria =
			requestEligibilityCriteriaService.retrieveRequestEligibilityCriteria(facilityId);

		if(requestEligibilityCriteria != null) {
			CustomRequestSettings customRequestSettings = requestEligibilityCriteria.getCustomRequestSettings();

			if(customRequestSettings != null) {
				for(CustomRequestSetting customRequestSetting : customRequestSettings) {
					if(customRequestSetting.getId().equals(appointmentRequest.getTypeOfCareId())) {
						return customRequestSetting;
					}
				}
			}
		}

		return null;
	}

	private void validateStatus(VARAppointmentRequest appointmentRequest) {
		String status = appointmentRequest.getStatus();

		if (NullChecker.isNotNullish(status)) {
			boolean found = false;
			AppointmentRequestStatus[] statuses = AppointmentRequestStatus.values();
			for (AppointmentRequestStatus statusEnum : statuses) {
				if (status.equals(statusEnum.getName())) {
					found = true;
					break;
				}
			}

			if (!found) {
				ValidationResult<VARAppointmentRequest> validationResult =
					createOpenAppointmentRequestValidationError(appointmentRequest, INVALID_STATUS_ERROR_MSG);

				throw new InvalidAppointmentRequestException(validationResult.getErrors());
			}
		} else {
			ValidationResult<VARAppointmentRequest> validationResult =
				createOpenAppointmentRequestValidationError(appointmentRequest, STATUS_NOT_FOUND_ERROR_MSG);

			throw new InvalidAppointmentRequestException(validationResult.getErrors());
		}
	}

	private ValidationResult<VARAppointmentRequest> createValidationError(VARAppointmentRequest appointmentRequest,
																		  String fieldName, String errorMessage) {
		ValidationResult<VARAppointmentRequest> validationResult = new ValidationResult<VARAppointmentRequest>();
		validationResult.setRequestObject(appointmentRequest);

		ValidationError error = new ValidationError();
		error.setErrorMessage(errorMessage);
		error.setFieldName(fieldName);

		validationResult.addIfNotNull(error);
		return validationResult;
	}

	private ValidationResult<VARAppointmentRequest> createOpenAppointmentRequestValidationError(VARAppointmentRequest appointmentRequest, String errorMessage) {
		return createValidationError(appointmentRequest, "error", errorMessage);
	}

    private void throwAnErrorIfAppointmentRequestDateHasPassed(VARAppointmentRequest appointmentRequest,
															   VARAppointmentRequestMessage appointmentRequestMessage) {

		if (!hasAppointmentDatePassed(appointmentRequest.getAppointmentDate())){
            ValidationResult<VARAppointmentRequestMessage> validationResult =
				createOpenAppointmentRequestMessageValidationError(appointmentRequestMessage,
					APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_AFTER_APPOINTMENT_DATE_PASSED);

            throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
        }
    }
    private boolean hasAppointmentDatePassed(String appointmentRequestDate) {

        if(NullChecker.isNotNullish(appointmentRequestDate)){
            Date today = DateHelper.beginOfDate(DateHelper.getToday());
            Date appointmentDate = DateHelper.beginOfDate(DateHelper.parseDate(appointmentRequestDate));
            if(today.after(appointmentDate))
                return false;
        }

        return true;
    }

    private void throwAnErrorIfAppointmentRequestIsCancelled(VARAppointmentRequest appointmentRequest,
															 VARAppointmentRequestMessage appointmentRequestMessage) {
        if (appointmentRequest.getStatus().equals(AppointmentRequestStatus.CANCELLED.getName())){
            ValidationResult<VARAppointmentRequestMessage> validationResult =
				createOpenAppointmentRequestMessageValidationError(appointmentRequestMessage,
					APPOINTMENT_REQUEST_MSG_NOT_ALLOWED_FOR_CANCELLED_APPOINTMENT);

            throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
        }
    }

    private ValidationResult<VARAppointmentRequestMessage> createOpenAppointmentRequestMessageValidationError(
		VARAppointmentRequestMessage appointmentRequestMessage, String errorMessage) {

        ValidationResult<VARAppointmentRequestMessage> validationResult = new ValidationResult<VARAppointmentRequestMessage>();
        validationResult.setRequestObject(appointmentRequestMessage);

        ValidationError error = new ValidationError();
        error.setErrorMessage(errorMessage);
        error.setFieldName("error");

        validationResult.addIfNotNull(error);
        return validationResult;
    }

    private void throwAnErrorIfMessageLimitReached(VARAppointmentRequest appointmentRequest,
												   VARAppointmentRequestMessage appointmentRequestMessage){

        VARAppointmentRequestMessages currentMessagesForAppointment =
			fetchAppointmentRequestMessages(appointmentRequest.getPatientIdentifier(),
				appointmentRequest.getDataIdentifier());

        int numberOfMessagesAlreadySentBySender = 0;
        for(VARAppointmentRequestMessage aptReqMsg : currentMessagesForAppointment){
            if(appointmentRequestMessage.getSenderId().equals(aptReqMsg.getSenderId())){
                numberOfMessagesAlreadySentBySender ++;
            }
        }

        if(numberOfMessagesAlreadySentBySender >= messageLimitPerSender){
            ValidationResult<VARAppointmentRequestMessage> validationResult =
				createOpenAppointmentRequestMessageValidationError(appointmentRequestMessage,
					APPOINTMENT_REQUEST_MSG_LIMIT_REACHED);

            throw new InvalidAppointmentRequestMessageException(validationResult.getErrors());
        }

    }

	private DateFilter createDateFilterWithDefaultDateRange() {
		Date endDate = DateHelper.getToday();
		Date startDate = DateHelper.beginOfDate(DateHelper.minusDays(endDate, FarthestTimeVeteranCanRequestAppointmentInDays));
		return DateFilterFactory.createFilterFromDate(startDate, endDate);
	}

	public void addSurrogate(PatientData patientData, VamfUser vamfUser) {
		if (vamfUser != null) {
			MhpUser user = vamfUser.getMhpUser();
			if(user != null && !user.userIsPatient()) {
				patientData.setSurrogateIdentifier("EDIPI", user.getId());
			}
		}
	}

	private boolean doesUserHasStaffRole(VamfUser vamfUser) {
		boolean staff = false;
		List<String> roles = vamfUser.getAuthorizedRoles();
		if (roles.contains(ROLE_STAFF)) {
			staff = true;
		}
		return staff;
	}

	private void validateEdipiPatientIdentifier(Patient patient, VARAppointmentRequest appointmentRequest) {
		//Check identifier on Patient first
		PatientIdentifier identifier = patient.getPatientIdentifier();
		if (identifier.getUniqueId() == null || !"EDIPI".equalsIgnoreCase(identifier.getAssigningAuthority())) {
			// EDIPI not found so use one pulled from MVI and passed on the request object
			PatientIdentifier requestIdentifier = appointmentRequest.getPatientIdentifier();
			patient.setPatientIdentifier(patientIdentifiersDataService.getTargetPatientIdentifier(requestIdentifier.getAssigningAuthority(), requestIdentifier.getUniqueId()));
		}
	}

	public PatientRequestLimit getPatientRequestLimit(PatientIdentifier patientIdentifier, String clinicalServiceId, String institutionCode) {
		PatientRequestLimit patientRequestLimit = new PatientRequestLimit();

    	final RequestEligibilityCriteria requestEligibilityCriteria = requestEligibilityCriteriaService.retrieveRequestEligibilityCriteria(institutionCode);
    	if (requestEligibilityCriteria != null) {
			requestEligibilityCriteria.getSetting(clinicalServiceId).ifPresent(new Consumer<CoreSetting>() { // TODO: switch to lambda
				@Override
				public void accept(final CoreSetting coreSetting) {
					patientRequestLimit.setRequestLimit(coreSetting.getSubmittedRequestLimit());
				}
			});
		}

		final PatientIdentifier targetIdentifier = patientIdentifiersDataService.getTargetPatientIdentifier(
				patientIdentifier.getAssigningAuthority(),
				patientIdentifier.getUniqueId()
		);

		final int requestCount = appointmentRequestDataLayer.getSubmittedRequestCountByApptType(
				targetIdentifier.getUniqueId(),
				createDateFilterWithDefaultDateRange(),
				clinicalServiceId,
				institutionCode
		);
		patientRequestLimit.setNumberOfRequests(requestCount);

		return patientRequestLimit;
	}

	/**
	 * Update appointment requests with mongo friendly name for all locations.
	 *
	 * @param appointmentRequests
	 */
	protected void updateAppointmentRequestsWithMongoFriendlyName(Collection<VARAppointmentRequest> appointmentRequests) {
		final Collection<String> institutionCodes = new HashSet<>();

		for (VARAppointmentRequest appointmentRequest : appointmentRequests) {
			if (NullChecker.isNotNullish(appointmentRequest.getFacility().getFacilityCode())) {
				institutionCodes.add(appointmentRequest.getFacility().getFacilityCode());
			}
		}

		final Map<String, CustomFriendlyText> customFriendlyTextMap =
				getCustomFriendlyTextMap(institutionCodes);

		for (VARAppointmentRequest appointmentRequest : appointmentRequests) {
			CustomFriendlyText customFriendlyText = customFriendlyTextMap.get(appointmentRequest.getFacility().getFacilityCode());

			if (customFriendlyText != null && NullChecker.isNotNullish(customFriendlyText.getFriendlyText())) {
				appointmentRequest.setFriendlyLocationName(customFriendlyText.getFriendlyText());
			}
		}
	}

	private Map<String, CustomFriendlyText> getCustomFriendlyTextMap(Collection<String> institutionCodes) {
		Map<String, CustomFriendlyText> customFriendlyTextHashMap =  new HashMap<>(institutionCodes.size());

		final CustomFriendlyTexts customFriendlyTexts =
				customFriendlyTextService.fetchByInstitutionCodes(institutionCodes);

		for (final CustomFriendlyText customFriendlyText : customFriendlyTexts) {
			customFriendlyTextHashMap.put(customFriendlyText.getSiteCode(), customFriendlyText);
		}

		return customFriendlyTextHashMap;
	}
}
